-- Cache Trimmer
-- Scans a dir tree, sorts by age, sums file sizes in age order,
-- and deletes all older files once a maximum size is reached.
-- Last modified: 24-Jun-11

-- Default args:
local MaxSize = 1024			-- Arg 1: Maximum dir tree size, in MB
local Path = "LoaderCache:"		-- Arg 2: Dir tree path (requires 'fusion' to be set)


function Say(str)
	print(str.."\n")
	if fusion and fusion.CurrentComp then
		fusion.CurrentComp:Print(str.."\n")
	end

end

-- GetDirTree()
-- Returns a flat table containing files from all subdirs
-- Format is like bmd.readdir(), but with an additional 
-- (absolute) Path entry for each file 
function GetDirTree(path, filelist)
	if filelist == nil then
		filelist = {}
	end

	local subdir = bmd.readdir(path.."*")
		
	for i,entry in ipairs(subdir) do
		if entry.IsDir then
			filelist = GetDirTree(path..entry.Name.."\\", filelist)
		else
			entry.Path = path
			table.insert(filelist, entry)
		end
	end
	return filelist
end

-- PruneDirTree()
-- Iterates subdirs, removing empty subfolders
-- Returns number of pruned dirs
function PruneDirTree(path)
	local dircount = 0
	local subdir = bmd.readdir(path.."*")

	if subdir[1] == nil then	-- no entries?
		bmd.removedir(path)
		dircount = 1
	else
		for i,entry in ipairs(subdir) do
			if entry.IsDir then
				dircount = dircount + PruneDirTree(path..entry.Name.."\\")
			end
		end
	end
	return dircount
end


-- Main code start --

-- Get args, if any
if arg and arg[1] ~= nil then
	MaxSize = tonumber(arg[1])
end

local path
if arg then
	path = arg[2]
end
if path == nil then
	if fusion ~= nil then
		path = fusion:MapPath(Path)
	else
		Say("No path given");
		exit(20)
	end
end
if string.sub(path, -1) ~= "\\" then
	path = path.."\\"
end

Say("Trimming "..path.." to less than ".. MaxSize.." MB...")

-- Get the full list of files
local filelist = GetDirTree(path)

-- Sort them by increasing age
table.sort(filelist, function(entry1, entry2) return entry1.WriteTime > entry2.WriteTime end)

-- Add sizes until we reach our limit, then start deleting
local filecount = 0
local dircount = 0
local delsize = 0
local size = 0
MaxSize = MaxSize * 1048576

for i,entry in ipairs(filelist) do
	size = size + entry.Size
	
	if size > MaxSize then
		ok, msg = os.remove(entry.Path..entry.Name)
--Say("Deleting "..entry.Path..entry.Name..", size: "..math.ceil(entry.Size / 1048576).." MB")

		if msg ~= nil then
			Say("Failed to delete: "..msg)
		else
			filecount = filecount + 1
			delsize = delsize + entry.Size
		end
	end
end

-- Now prune empty dirs
local maindir = bmd.readdir(path.."*")
for i,entry in ipairs(maindir) do
	if entry.IsDir then
		dircount = dircount + PruneDirTree(path..entry.Name.."\\")
	end
end

Say("Deleted "..filecount.." files and "..dircount.." dirs, totalling "..math.ceil(delsize / 1048576).." MB")
